<?php

namespace Omnipay\WorldPay;

use Omnipay\Tests\GatewayTestCase;

class GatewayTest extends GatewayTestCase {

    public function setUp() {
        parent::setUp();

        $this->gateway = new Gateway($this->getHttpClient(), $this->getHttpRequest());
        $this->gateway->setCallbackPassword('bar123');

        $this->options = array(
            'amount' => '10.00',
            'returnUrl' => 'https://www.example.com/return',
        );
    }

    public function testPurchase() {
        $response = $this->gateway->purchase($this->options)->send();

        $this->assertFalse($response->isSuccessful());
        $this->assertTrue($response->isRedirect());
        $this->assertNull($response->getTransactionReference());
        $this->assertContains('https://secure.worldpay.com/wcc/purchase?', $response->getRedirectUrl());
    }

    public function testCompletePurchaseSuccess() {
        $this->getHttpRequest()->request->replace(
                array(
                    'callbackPW' => 'bar123',
                    'transStatus' => 'Y',
                    'transId' => 'abc123',
                    'rawAuthMessage' => 'hello',
                )
        );

        $response = $this->gateway->completePurchase($this->options)->send();

        $this->assertTrue($response->isSuccessful());
        $this->assertFalse($response->isRedirect());
        $this->assertEquals('abc123', $response->getTransactionReference());
        $this->assertSame('hello', $response->getMessage());
    }

    /**
     * @expectedException \Omnipay\Common\Exception\InvalidResponseException
     */
    public function testCompletePurchaseInvalidCallbackPassword() {
        $this->getHttpRequest()->request->replace(
                array(
                    'callbackPW' => 'fake',
                )
        );

        $response = $this->gateway->completePurchase($this->options)->send();
    }

    public function testCompletePurchaseError() {
        $this->getHttpRequest()->request->replace(
                array(
                    'callbackPW' => 'bar123',
                    'transStatus' => 'N',
                    'rawAuthMessage' => 'Declined',
                )
        );

        $response = $this->gateway->completePurchase($this->options)->send();

        $this->assertFalse($response->isSuccessful());
        $this->assertFalse($response->isRedirect());
        $this->assertNull($response->getTransactionReference());
        $this->assertSame('Declined', $response->getMessage());
    }

}
